unit threadedICYClient;

{* Delphi Streaming Radio Library
 * Copyright 2004-2007, Steve Blinch
 * http://code.blitzaffe.com
 * ============================================================================
 *
 * LICENSE
 *
 * This code is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *	
 * You should have received a copy of the GNU General Public License along
 * with this code; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *}

interface

uses Classes,IdTCPClient,IdTCPConnection,threadLogUnit;

const
  DEFAULT_ICY_PORT = 8000;

  MSG_ICYCLIENT_FREE = 2001;
  MSG_ICYCLIENT_CONNECTED = 2002;
  MSG_ICYCLIENT_STREAMING = 2003;
  MSG_ICYCLIENT_BUFFER = 2004;
  MSG_ICYCLIENT_METADATA = 2005;

  THREADCLIENT_CONNECT_TIMEOUT = 4000;

const
  BUFFER_SIZE = 4096;
Type

  TStreamInfo = record
      Bitrate: Integer;
      Desc,
      Genre,
      Name,
      Pub,
      URL,
      ContentType: String;
    end;
  TMetaState = (msLengthByte,msReadData,msNone);

  TBuffer = array[0..BUFFER_SIZE-1] of byte;
  PBuffer = ^TBuffer;
  TThreadedICYClient = class(TThread)
    private
      ThreadLog: TThreadLog;
      Connection: TIdTCPConnection;

      DataBuffer: PBuffer;
      DataCursor: Integer; // current position in the buffer, used by BufReadLn
      DataSize: Integer;   // current size of data in the buffer
      Negotiating: Boolean;

      MetaInterval: Integer;
      MetaBytes: Integer;
      MetaRemaining: Integer;
      MetaString: String;

      StreamInfo: TStreamInfo;

      MetaState: TMetaState;

      PendingLF: Boolean;     // returns TRUE if the last buffer ended on a carriage return
      BufReadLnCache: String; // contains buffered (incomplete) data from last BufReadLn() call

      ReceivedHTTPResponse: Boolean;

      TCPClient: TIdTCPClient;

      function CheckSocket: Boolean;
      function Negotiate: Boolean;

      procedure ReturnMPEGData(Buf: PBuffer; Size: Integer);
      function BufReadLn(var S: String): Boolean;

      function ProcessHeader(Header: String): boolean;

    public
      MountPoint: String;
      Host: String;
      Port: Integer;
      Timeout: Integer;
      WindowHandle: THandle;

      constructor Create(Suspended: Boolean; URI: String);
      destructor Destroy; override;

      procedure Execute; override;
    end;

implementation

uses //peerConnectionUnit,
Messages,Windows,SysUtils,IdException,wmCommUnit,strUtils,rawbufferUnit;

function rpos(C: Char; S: String): Integer;
var i: Integer;
begin
  i:=Length(S);
  while ((i>0) and (S[i]<>C)) do Dec(i);
  Result:=i;
end;

function extractMountPoint(var Host: String): String;
var p: Integer;
begin
  Result:='';
  p:=rpos('/',Host);
  if (p>0) then
    begin
      Result:=Copy(Host,p,Length(Host)-p+1);
      SetLength(Host,p-1);
    end
end;

constructor TThreadedICYClient.Create(Suspended: Boolean; URI: String);
var
  p: Integer;
  Hostname: String;
begin
  Timeout:=THREADCLIENT_CONNECT_TIMEOUT;

  if (Copy(URI,1,6)='icy://') then URI:=Copy(URI,6+1,Length(URI)-6+1);
  MountPoint:=extractMountPoint(URI);

  p:=Pos(':',URI);
  if (p>0) then
    begin
      Hostname:=Copy(URI,1,p-1);
      Delete(URI,1,p);
      Host:=Hostname;
      Port:=StrToIntDef(URI,DEFAULT_ICY_PORT);
    end
    else
    begin
      Host:=URI;
      Port:=DEFAULT_ICY_PORT;
    end;

  inherited Create(Suspended);
end;

destructor TThreadedICYClient.Destroy;
begin
  inherited;
end;

procedure TThreadedICYClient.Execute;
var
//  Peer: TPeerConnection;
  OK: Boolean;
begin
  try
    ThreadLog:=TThreadLog.Create(WindowHandle,Self);
    ThreadLog.Log('ICY thread startup ('+Host+')',LL_NORMAL);

    try
      TCPClient:=TIdTCPClient.Create(nil);
      TCPClient.Host:=Host;
      TCPClient.Port:=Port;

      try
        TCPClient.Connect(Timeout);
      except
        on E: Exception do
          begin
            ThreadLog.Log('ICY connection failed ('+Host+': '+E.Message+')',LL_NORMAL);

            // the finally statements handle cleanup here, so just exit
            exit;
          end;
      end;

      PostMessage(WindowHandle,WM_USER,MSG_ICYCLIENT_CONNECTED,0);

      Connection:=TCPClient;

      TCPClient.WriteLn(Format('GET %s HTTP/1.0',[MountPoint]));
      TCPClient.WriteLn('Icy-MetaData:1');
      TCPClient.WriteLn('');

      ReceivedHTTPResponse:=False;
      Negotiating:=True;
      BufReadLnCache:='';
      PendingLF:=False;
      MetaBytes:=0;
      MetaInterval:=-1;
      MetaString:='';
      MetaState:=msNone;

      try
        New(DataBuffer);

        while not Terminated do
          begin
            if (Negotiating) then
              OK:=Negotiate
            else
              OK:=CheckSocket;

            if not OK then break;
          end;
      finally
        Dispose(DataBuffer);
      end;

    finally
      try
        TCPClient.Disconnect;
      except
        on E: EIdConnClosedGracefully do
        ; // ignore EIdConnClosedGracefully
        on E: Exception do ;
      end;
      TCPClient.Free;

    end;

  finally
    {
    // apparently we can't call or free ThreadLog here without causing an
    // intermittant AV at 00404218 or 00001F48... why??
    ThreadLog.Log('ICY thread shutdown ('+Host+')',LL_NORMAL);
    ThreadLog.Free;
      }
    SendMessage(WindowHandle,WM_USER,MSG_ICYCLIENT_FREE,Integer(Self));
  end;
end;

function TThreadedICYClient.BufReadLn(var S: String): Boolean;
var
  i: Integer;
  Copied: String;
  BytesCopied: Integer;
begin
  i:=DataCursor;
  if (PendingLF) then
    begin
      PendingLF:=False;
      Inc(i);
    end;
  while (i<DataSize) and (DataBuffer^[i]<>13) do
    Inc(i);

  BytesCopied:=i-DataCursor;
  SetLength(Copied,BytesCopied);
  Move(DataBuffer^[DataCursor],Copied[1],BytesCopied);
  Inc(DataCursor,BytesCopied+1);

  if (DataCursor=DataSize) and (DataBuffer^[i]=13) then PendingLF:=True;
//  ThreadLog.Log('Data cursor = '+IntToStr(DataCursor)+'; data size = '+IntToStr(DataSize),LL_NORMAL);

//  ThreadLog.Log('DataBuffer^[i]=#'+IntToStr(DataBuffer^[i]),LL_NORMAL);

  if (DataBuffer^[i]=13) and (not PendingLF) then
    begin
      Inc(DataCursor);

      S:=BufReadLnCache+Copied;
      BufReadLnCache:='';
      Result:=True;

    end
    else
    begin
      BufReadLnCache:=BufReadLnCache+Copied;
      Result:=False;
    end;
end;

function TThreadedICYClient.Negotiate: Boolean;
var
  i: Integer;
  S: String;
  InfoBuffer: TRawBuffer;
begin
  Connection.ReadFromStack(True,1,False);
  Result:=Connection.Connected;

  // did we receive data from the connected host?
  if (Connection.InputBuffer.Size>0) then
    begin
      DataSize:=Connection.InputBuffer.Size;
      if (DataSize>BUFFER_SIZE) Then DataSize:=BUFFER_SIZE;

      DataCursor:=0;
      Connection.ReadBuffer(DataBuffer^,DataSize);

      while (BufReadLn(S)) do
        begin
          if (S='') then  // WOOT!  empty string means MPEG data starts now!
            begin
              Negotiating:=False;
              Result:=True;


              InfoBuffer:=TRawBuffer.Create(SizeOf(TStreamInfo));
              InfoBuffer.Store(StreamInfo,SizeOf(TStreamInfo));

              wmPostBuffer(WindowHandle,MSG_ICYCLIENT_STREAMING,InfoBuffer);

              ReturnMPEGData(Pointer(Integer(DataBuffer)+DataCursor),DataSize-DataCursor);
              break;
            end
            else
            begin
              Result:=ProcessHeader(S);
              if (not Result) then break;
            end;
        end;
    end;
end;

function extractStreamTitle(MetaString: String): String;
var
  p: Integer;
  i: Integer;
begin
  p:=pos('''',MetaString);
  Delete(MetaString,1,p);
  p:=pos('''',MetaString);
  Result:=Copy(MetaString,1,p-1);
end;

function TThreadedICYClient.CheckSocket: Boolean;
var
  i: Integer;
  S: String;
begin
  Connection.ReadFromStack(True,1,False);
  Result:=Connection.Connected;

  // did we receive data from the connected host?
  if (Connection.InputBuffer.Size>0) then
    begin
      DataSize:=Connection.InputBuffer.Size;

      if (DataSize>BUFFER_SIZE) Then DataSize:=BUFFER_SIZE;

      if (MetaInterval>0) and (MetaBytes=MetaInterval) then
        begin
          case MetaState of
            msNone:
              begin
                DataSize:=1;
                MetaState:=msLengthByte;
                MetaString:='';
              end;

              msLengthByte,
              msReadData:
                begin
                  DataSize:=MetaRemaining;
                end;
            end;

        end
        else
        begin
          if (MetaBytes+DataSize>MetaInterval) then
            begin
{
              ThreadLog.Log(
                Format(
                  'Buffer would exceed boundary, so limiting: MetaBytes=%d+DataSize=%d>MetaInterval=%d, setting DataSize=%d',
                  [MetaBytes,DataSize,MetaInterval,MetaInterval-MetaBytes]
                ),
                LL_NORMAL
              );
 }
              DataSize:=MetaInterval-MetaBytes;
            end;
        end;

        Connection.ReadBuffer(DataBuffer^,DataSize);

        case MetaState of
            msNone:
              ReturnMPEGData(DataBuffer,DataSize);
            msLengthByte:
              begin
                MetaState:=msReadData;
                MetaRemaining:=DataBuffer^[0]*16;
              end;
            msReadData:
              begin
                SetLength(S,DataSize);
                Move(DataBuffer^[0],S[1],DataSize);
                MetaString:=MetaString+S;
                Dec(MetaRemaining,DataSize);
                
                if (MetaRemaining=0) then
                  begin
                    if (Length(MetaString)>0) then
                      wmSendString(WindowHandle,MSG_ICYCLIENT_METADATA,extractStreamTitle(MetaString));

                    MetaBytes:=0;
                    MetaState:=msNone;
                  end;
              end;
          end;
    end;
end;

procedure TThreadedICYClient.ReturnMPEGData(Buf: PBuffer; Size: Integer);
begin
  if (Size>0) then
    begin
      Inc(MetaBytes,Size);

      //ThreadLog.Log('STAT ICY uplink processing '+IntToStr(Size)+'-byte buffer',LL_NORMAL);

      // send a copy of Buf to the main thread (posted to avoid blocking)
      wmPostBuffer(WindowHandle,MSG_ICYCLIENT_BUFFER,Buf,Size);
    end;
end;

procedure split(c: Char; s: String; var s1,s2: String);
var p: Integer;
begin
  p:=Pos(C,S);
  S1:=Copy(S,1,p-1);
  S2:=Copy(S,p+1,Length(S)-p);
end;

function TThreadedICYClient.ProcessHeader(Header: String): boolean;
var Key,Value: String;
begin
  Result:=True;
  if (Header='') then exit;

  // make sure we received a valid HTTP/1.0 200 OK response
  if (not ReceivedHTTPResponse) then
    begin
      ReceivedHTTPResponse:=True;

      Result:=(Header='HTTP/1.0 200 OK');
      if (not Result) then ThreadLog.Log('Invalid HTTP response: '+Header,LL_NORMAL);
      exit;
    end;

  split(':',Header,Key,Value);
  ThreadLog.Log(Key+': '+Value,LL_NORMAL);
  Key:=LowerCase(Key);

  if (Key='icy-metaint') then
    begin
      MetaInterval:=StrToIntDef(Value,-1);
      ThreadLog.Log('Metadata interval = '+IntToStr(MetaInterval),LL_NORMAL);
    end;
  if (Key='icy-br') then StreamInfo.BitRate:=StrToIntDef(Value,128);
  if (Key='icy-description') then StreamInfo.Desc:=Value;
  if (Key='icy-genre') then StreamInfo.Genre:=Value;
  if (Key='icy-name') then StreamInfo.Name:=Value;
  if (Key='icy-pub') then StreamInfo.Pub:=Value;
  if (Key='icy-url') then StreamInfo.URL:=Value;
end;


end.
